# makro/event

Ein **leichtgewichtiges, PSR-14-kompatibles** Event-Paket für PHP 8: minimaler `EventDispatcher`,
ein einfacher `ListenerProvider` mit Prioritäten, sowie Utilities für Tests und stoppbare Events.
Keine weiteren Abhängigkeiten.

## Features

* ✅ **PSR-14**: Implementiert `EventDispatcherInterface` und `ListenerProviderInterface`.
* 🧩 **Prioritäten**: Höhere Priorität wird zuerst ausgeführt.
* 🧬 **Vererbungs-Matching**: Listener können auf **exakte Klassen**, **Elternklassen** und **Interfaces** hören.
* ⛔ **Stoppable Events**: Unterstützung über `StoppableEvent`/`StoppableEventTrait`.
* 💤 **Null-Overhead**: `NullEventDispatcher` für No-Op-Szenarien.

---

## Installation

```bash
composer require makro/event
```

> Anforderungen: PHP ^8.0
> `psr/event-dispatcher` ^1.0

---

## Schnellstart

```php
use Makro\Event\ListenerProvider;
use Makro\Event\EventDispatcher;

final class UserRegistered
{
    public function __construct(public int $userId) {}
}

$provider = new ListenerProvider();

// Listener registrieren (mit Priorität)
$provider->listen(UserRegistered::class, function (UserRegistered $e) {
    // z. B. Mail versenden
}, 50);

$events = new EventDispatcher($provider);

// Event auslösen
$events->dispatch(new UserRegistered(123));
```

### Vererbungs-/Interface-Matching

```php
interface DomainEvent {}
class BaseEvent implements DomainEvent {}
class ChildEvent extends BaseEvent {}

$provider->listen(BaseEvent::class,  fn(BaseEvent  $e) => /* ... */);
$provider->listen(DomainEvent::class, fn(DomainEvent $e) => /* ... */);

// Ruft beide Listener auf:
$events->dispatch(new ChildEvent());
```

### Stoppable Events

```php
use Makro\Event\StoppableEvent;

final class GateChecked extends StoppableEvent
{
    public function __construct(public string $route) {}
}

$provider->listen(GateChecked::class, function (GateChecked $e) {
    // stoppe weitere Listener (z. B. wegen fehlender Berechtigung)
    $e->stopPropagation();
}, 100);

$provider->listen(GateChecked::class, function (GateChecked $e) {
    // wird wegen stopPropagation() nicht mehr aufgerufen
});

$events->dispatch(new GateChecked('/admin'));
```

---

## Integration in einen konkreten Handler (Beispiel: `CopySurveyCommandHandler`)

**Ziel:** Nach erfolgreichem Kopieren einer Umfrage ein `SurveyCopiedEvent` auslösen,
damit optionale Listener (Aktivierung, Audit, Cache-Warming …) reagieren können – ohne `makro/command` zu verändern.

### 1) Event-DTO (reines Datenobjekt)

```php
namespace Makro\CopySurvey\Event;

final class SurveyCopiedEvent
{
    /** @param array<int,int> $questionIdMap altId => neuId */
    public function __construct(
        public int $sourceSurveyId,
        public int $newSurveyId,
        public array $questionIdMap = [],
        public ?string $actorId = null,
        public ?string $correlationId = null,
        public \DateTimeImmutable $occurredAt = new \DateTimeImmutable(),
    ) {}
}
```

### 2) Handler: Dispatcher injizieren & Event dispatchen

```php
namespace Makro\CopySurvey\CommandHandler;

use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Result\CommandHandlerResult;
use Makro\CopySurvey\Command\CopySurveyCommand;
use Makro\CopySurvey\Event\SurveyCopiedEvent;
use Psr\EventDispatcher\EventDispatcherInterface;
use Makro\Event\NullEventDispatcher;

final class CopySurveyCommandHandler extends AbstractCommandHandler
{
    private EventDispatcherInterface $events;

    public function __construct(
        ?EventDispatcherInterface $events = null,
        ?\Makro\Command\Contract\CommandHandlerGuardInterface $handlerGuard = null,
        ?\Makro\Command\Contract\CommandGuardInterface        $commandGuard = null,
    ) {
        $this->events = $events ?? new NullEventDispatcher(); // optional, kein Zwang zu Events
        parent::__construct($handlerGuard, $commandGuard);
    }

    /** @param CopySurveyCommand $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {
        // 1) Kopierlogik (vereinfacht)
        $newSurveyId = 12345;
        $qMap = [10 => 210, 11 => 211];

        // 2) Event NACH Erfolg auslösen
        $this->events->dispatch(new SurveyCopiedEvent(
            sourceSurveyId: $command->sourceSurveyId,
            newSurveyId:    $newSurveyId,
            questionIdMap:  $qMap,
            actorId:        $command->actorId ?? null,
            correlationId:  $command->correlationId ?? null,
        ));

        // 3) Ergebnis wie gewohnt
        return CommandHandlerResult::success([
            'sourceSurveyId' => $command->sourceSurveyId,
            'newSurveyId'    => $newSurveyId,
            'questionIdMap'  => $qMap,
        ], 'Survey copied.');
    }

    protected function supportedCommand(): string
    {
        return \Makro\CopySurvey\Command\CopySurveyCommand::class;
    }
}
```

### 3) Wiring ohne Framework

```php
use Makro\Event\ListenerProvider;
use Makro\Event\EventDispatcher;
use Makro\CopySurvey\Event\SurveyCopiedEvent;

$provider = new ListenerProvider();

// Listener registrieren
$provider->listen(SurveyCopiedEvent::class, function (SurveyCopiedEvent $e): void {
    // z. B. frisch kopierte Umfrage aktivieren oder Audit schreiben
}, 100);

$events = new EventDispatcher($provider);

// Dispatcher dem Handler übergeben
$handler = new \Makro\CopySurvey\CommandHandler\CopySurveyCommandHandler($events);

// $handler->handle(new CopySurveyCommand(...));
```

> **Hinweis zu Transaktionen:** Wenn das Kopieren in einer DB-Transaktion läuft, dispatch nach dem Commit – oder halte Listener idempotent.

---

## API-Übersicht

### `Makro\Event\ListenerProvider`

* `listen(string $eventClass, callable $listener, int $priority = 0): void`
  Registriert Listener für eine Eventklasse/-Schnittstelle. Höhere `priority` wird zuerst aufgerufen.
  Listener-Auflösung: **exakte Klasse → Elternklassen → Interfaces**.

* `getListenersForEvent(object $event): iterable<callable>`
  (PSR-14) Liefert die passenden Listener.

### `Makro\Event\EventDispatcher`

* `dispatch(object $event): object`
  (PSR-14) Ruft alle passenden Listener in Reihenfolge auf.
  Stoppt bei `StoppableEventInterface::isPropagationStopped()`.

### `Makro\Event\NullEventDispatcher`

* No-Op-Dispatcher für Fälle, in denen Events **optional** sein sollen.

### `Makro\Event\RecordingEventDispatcher`

* Nimmt alle Events in `public array $events` auf – nützlich in Tests.

### `Makro\Event\StoppableEvent` / `Makro\Event\StoppableEventTrait`

* Basis-Klasse bzw. Trait zum einfachen Implementieren stoppbarer Events.

---

## Testen (Beispiele mit Pest)

**Prioritäten:**

```php
$provider = new ListenerProvider();
$log = [];

class E {}
$provider->listen(E::class, fn(E $e) => $log[]='p10', 10);
$provider->listen(E::class, fn(E $e) => $log[]='p50', 50);
$provider->listen(E::class, fn(E $e) => $log[]='p0',   0);

(new EventDispatcher($provider))->dispatch(new E());

expect($log)->toEqual(['p50','p10','p0']);
```

**Stoppable:**

```php
final class StopMe extends \Makro\Event\StoppableEvent {}

$hits = 0;
$provider->listen(StopMe::class, function (StopMe $e) use (&$hits) { $hits++; $e->stopPropagation(); }, 10);
$provider->listen(StopMe::class, function (StopMe $e) use (&$hits) { $hits++; }, 0);

(new EventDispatcher($provider))->dispatch(new StopMe());

expect($hits)->toBe(1);
```

**Recording:**

```php
$rec = new \Makro\Event\RecordingEventDispatcher();
$rec->dispatch(new stdClass());
expect($rec->events)->toHaveCount(1);
```

---

## Design-Entscheidungen & Hinweise

* **Fail-fast**: Listener-Exceptions werden **durchgereicht**. Für „weiches“ Verhalten kannst du lokal `try/catch` verwenden oder eine separate, eigene `SafeEventDispatcher`-Variante implementieren.
* **Determinismus**: Reihenfolge basiert auf Priorität (höher zuerst). Bei gleicher Priorität ist die Aufrufreihenfolge die Registrierungsreihenfolge (FIFO) *sofern entsprechend implementiert*; andernfalls registriere unterschiedliche Prioritäten.
* **Kopplung**: Events als **DTOs** halten (keine Logik), damit Listener frei kombinierbar sind.
* **Optionalität**: Durch `NullEventDispatcher` bleibt dein Code unabhängig von Event-Wiring.

---

## Changelog (Kurz)

* **v1.0.0**: Erste Veröffentlichung – `ListenerProvider`, `EventDispatcher`, `NullEventDispatcher`, `StoppableEvent`/`StoppableEventTrait`.
