<?php

declare(strict_types=1);

/**
 * Fixtures für Batch-Tests:
 * - DemoItemCommand (CommandInterface)
 * - DemoItemCommandHandler (AbstractCommandHandler<DemoItemCommand>)
 * Der Handler liefert success/failure/exception je nach Flags im Command.
 */
namespace Tests\Fixtures\Batch\Command {
    use Makro\Command\Contract\CommandInterface;

    final class DemoItemCommand implements CommandInterface
    {
        public function __construct(
            public int $n,
            public bool $fail = false,
            public bool $crash = false,
        ) {}
    }
}

namespace Tests\Fixtures\Batch\CommandHandler {
    use Makro\Command\CommandHandler\AbstractCommandHandler;
    use Makro\Command\Contract\CommandInterface;
    use Makro\Command\Result\CommandHandlerResult;
    use Tests\Fixtures\Batch\Command\DemoItemCommand;

    /** @extends AbstractCommandHandler<DemoItemCommand> */
    final class DemoItemCommandHandler extends AbstractCommandHandler
    {
        /** @param DemoItemCommand $command */
        protected function doHandle(CommandInterface $command): CommandHandlerResult
        {
            /** @var DemoItemCommand $command */
            if ($command->crash) {
                throw new \RuntimeException('crash');
            }
            if ($command->fail) {
                return CommandHandlerResult::failure('boom', ['n' => $command->n]);
            }
            return CommandHandlerResult::success(['n' => $command->n]);
        }
    }
}

/**
 * Tests für Makro\Command\CommandHandler\BatchCommandHandler
 */
namespace Tests\Core\Batch {
    use Makro\Command\Command\BatchCommand;
    use Makro\Command\CommandHandler\BatchCommandHandler;
    use Tests\Fixtures\Batch\Command\DemoItemCommand;
    use Tests\Fixtures\Batch\CommandHandler\DemoItemCommandHandler;

    it('processes a happy batch (all succeed)', function () {
        $items = [
            new DemoItemCommand(1),
            new DemoItemCommand(2),
        ];

        $batch   = new BatchCommand($items, stopOnFirstError: false);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler());

        $res = $handler->handle($batch);

        expect($res->success)->toBeTrue()
            ->and($res->data['total'])->toBe(2)
            ->and($res->data['successes'])->toBe(2)
            ->and($res->data['failures'])->toBe(0)
            ->and($res->data['items'][0]['status'])->toBe('success')
            ->and($res->data['items'][1]['status'])->toBe('success');
    });

    it('aggregates failure and continues when stopOnFirstError=false', function () {
        $items = [
            new DemoItemCommand(1),
            new DemoItemCommand(2, fail: true), // failure
            new DemoItemCommand(3),
        ];

        $batch   = new BatchCommand($items, stopOnFirstError: false);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler());

        $res = $handler->handle($batch);

        expect($res->success)->toBeFalse()
            ->and($res->data['total'])->toBe(3)
            ->and($res->data['successes'])->toBe(2)
            ->and($res->data['failures'])->toBe(1)
            ->and($res->data['items'][1]['status'])->toBe('failure')
            ->and($res->message)->toContain('1 failure');
    });

    it('stops on first failure when stopOnFirstError=true', function () {
        $items = [
            new DemoItemCommand(1),
            new DemoItemCommand(2, fail: true), // failure -> stop
            new DemoItemCommand(3),             // should not be processed
        ];

        $batch   = new BatchCommand($items, stopOnFirstError: true);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler());

        $res = $handler->handle($batch);

        expect($res->success)->toBeFalse()
            ->and($res->data['total'])->toBe(2) // nur 2 Items verarbeitet
            ->and($res->data['failures'])->toBe(1)
            ->and(count($res->data['items']))->toBe(2);
    });

    it('records exceptions from inner handler as error items', function () {
        $items = [
            new DemoItemCommand(1),
            new DemoItemCommand(99, crash: true), // throws RuntimeException
            new DemoItemCommand(2),
        ];

        $batch   = new BatchCommand($items, stopOnFirstError: false);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler());

        $res = $handler->handle($batch);

        expect($res->success)->toBeFalse()
            ->and($res->data['failures'])->toBe(1)
            ->and($res->data['items'][1]['status'])->toBe('error')
            ->and($res->data['items'][1]['exception'])->toBe(\RuntimeException::class);
    });

    it('throws when a batch item is not of the expected command type', function () {
        // Fake Command eines anderen Typs:
        $wrong = new class implements \Makro\Command\Contract\CommandInterface {};

        $items = [
            new DemoItemCommand(1),
            $wrong, // falscher Typ
        ];

        $batch   = new BatchCommand($items);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler());

        expect(fn() => $handler->handle($batch))
            ->toThrow(\InvalidArgumentException::class);
    });

    it('accepts explicit expectedClass override', function () {
        $items = [ new DemoItemCommand(7) ];

        $batch   = new BatchCommand($items);
        $handler = new BatchCommandHandler(new DemoItemCommandHandler(), expectedClass: DemoItemCommand::class);

        $res = $handler->handle($batch);

        expect($res->success)->toBeTrue()
            ->and($res->data['expected'])->toBe(DemoItemCommand::class);
    });
}
