<?php

declare(strict_types=1);

namespace Tests\Core\AbstractCommandHandler;

use LogicException;
use InvalidArgumentException;
use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Command\Contract\CommandInterface;

// 1) Kein \CommandHandler\ im FQCN
namespace Tests\Bad\NoMarker;
use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Command\Contract\CommandInterface;

final class OddHandler extends AbstractCommandHandler {
    protected function doHandle(CommandInterface $command): CommandHandlerResult {
        return CommandHandlerResult::success();
    }
}

// 2) Marker da, aber Klassenname endet nicht mit CommandHandler
namespace Tests\Bad\CommandHandler;
use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Command\Contract\CommandInterface;

final class Weird extends AbstractCommandHandler {
    protected function doHandle(CommandInterface $command): CommandHandlerResult {
        return CommandHandlerResult::success();
    }
}

// 3) Abgeleiteter Command existiert nicht
namespace Tests\Bad\CommandHandler;
final class GhostCommandHandler extends \Makro\Command\CommandHandler\AbstractCommandHandler {
    protected function doHandle(\Makro\Command\Contract\CommandInterface $command): \Makro\Command\Result\CommandHandlerResult {
        return \Makro\Command\Result\CommandHandlerResult::success();
    }
}

// 4) Abgeleiteter Command existiert, implementiert aber CommandInterface NICHT
namespace Tests\Bad;
final class NotACommand {}
namespace Tests\Bad\CommandHandler;
final class NotACommandHandler extends \Makro\Command\CommandHandler\AbstractCommandHandler {
    protected function doHandle(\Makro\Command\Contract\CommandInterface $command): \Makro\Command\Result\CommandHandlerResult {
        return \Makro\Command\Result\CommandHandlerResult::success();
    }
}

// 5) Falscher Command-Typ zur Laufzeit
namespace Tests\Bad\Runtime;
use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Command\Contract\CommandInterface;

final class GoodCommand implements CommandInterface {}
final class GoodCommandHandler extends AbstractCommandHandler {
    /** @return class-string<CommandInterface> */ // überschreiben, um Laufzeit zu erreichen
    protected function supportedCommand(): string { return GoodCommand::class; }
    protected function doHandle(CommandInterface $command): CommandHandlerResult { return CommandHandlerResult::success(); }
}
final class OtherCommand implements CommandInterface {}


// ==== Tests ====
namespace Tests\Core\AbstractCommandHandler;

use Tests\Bad\NoMarker\OddHandler;
use Tests\Bad\CommandHandler\Weird;
use Tests\Bad\CommandHandler\GhostCommandHandler;
use Tests\Bad\CommandHandler\NotACommandHandler;
use Tests\Bad\Runtime\GoodCommandHandler;
use Tests\Bad\Runtime\OtherCommand;

it('fails when \\CommandHandler\\ marker is missing', function () {
    $h = new OddHandler();
    expect(fn() => $h->handle(new class implements \Makro\Command\Contract\CommandInterface{}))
        ->toThrow(\LogicException::class);
});

it('fails when class name does not end with CommandHandler', function () {
    $h = new Weird();
    expect(fn() => $h->handle(new class implements \Makro\Command\Contract\CommandInterface{}))
        ->toThrow(\LogicException::class);
});

it('fails when inferred command class is missing', function () {
    $h = new GhostCommandHandler();
    expect(fn() => $h->handle(new class implements \Makro\Command\Contract\CommandInterface{}))
        ->toThrow(\LogicException::class);
});

it('fails when inferred command is not a CommandInterface', function () {
    $h = new \Tests\Bad\CommandHandler\NotACommandHandler();
    // irgendein Command-Objekt, damit handle() losläuft (wird vor doHandle abgewiesen)
    $dummyCmd = new class implements \Makro\Command\Contract\CommandInterface {};
    expect(fn() => $h->handle($dummyCmd))
        ->toThrow(\LogicException::class);
});

it('fails with InvalidArgumentException for wrong runtime command', function () {
    $h = new GoodCommandHandler();
    expect(fn() => $h->handle(new OtherCommand()))
        ->toThrow(\InvalidArgumentException::class);
});
