<?php

declare(strict_types=1);

namespace Makro\Command\CommandHandler;

use InvalidArgumentException;
use Makro\Command\Command\BatchCommand;
use Makro\Command\Contract\CommandGuardInterface;
use Makro\Command\Contract\CommandHandlerGuardInterface;
use Makro\Command\Contract\CommandHandlerInterface;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Result\CommandHandlerResult;

/**
 * @template T of CommandInterface
 * @extends AbstractCommandHandler<BatchCommand<T>>
 */
final class BatchCommandHandler extends AbstractCommandHandler
{
    /** @var CommandHandlerInterface<T> */
    private CommandHandlerInterface $inner;

    /** @var class-string<T> */
    private string $expected;

    /**
     * @param CommandHandlerInterface<T> $inner
     * @param class-string<T>|null $expectedClass   Falls null, wird aus $inner per Konvention abgeleitet.
     */
    public function __construct(
        CommandHandlerInterface $inner,
        ?string $expectedClass = null,
        ?CommandHandlerGuardInterface $handlerGuard = null,
        ?CommandGuardInterface $commandGuard = null,
    ) {
        $this->inner    = $inner;
        $this->expected = $expectedClass ?? $this->inferExpectedFromInner($inner);
        parent::__construct($handlerGuard, $commandGuard);
    }

    /** @return class-string<BatchCommand<T>> */
    protected function supportedCommand(): string
    {
        // Konventionsableitung passt hier nicht (BatchCommand liegt im Sub-Namespace).
        return BatchCommand::class;
    }

    /** @param BatchCommand<T> $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {
        /** @var BatchCommand<T> $command */
        $items = [];
        $successes = 0;
        $failures  = 0;

        foreach ($command->commands as $index => $item) {
            if (!$item instanceof $this->expected) {
                throw new InvalidArgumentException(
                    "commands[{$index}] must be instance of {$this->expected}, got " . $item::class
                );
            }

            try {
                $res = $this->inner->handle($item);

                if ($res->success) {
                    $successes++;
                    $items[] = [
                        'index'   => $index,
                        'status'  => 'success',
                        'data'    => $res->data,
                        'message' => $res->message,
                    ];
                } else {
                    $failures++;
                    $items[] = [
                        'index'   => $index,
                        'status'  => 'failure',
                        'data'    => $res->data,
                        'message' => $res->message,
                    ];
                    if ($command->stopOnFirstError) {
                        break;
                    }
                }
            } catch (\Throwable $e) {
                $failures++;
                $items[] = [
                    'index'     => $index,
                    'status'    => 'error',
                    'message'   => $e->getMessage(),
                    'exception' => $e::class,
                    'code'      => $e->getCode(),
                ];
                if ($command->stopOnFirstError) {
                    break;
                }
            }
        }

        $total   = $successes + $failures;
        $summary = [
            'total'            => $total,
            'successes'        => $successes,
            'failures'         => $failures,
            'stopOnFirstError' => $command->stopOnFirstError,
            'items'            => $items,
            'expected'         => $this->expected,
        ];

        return $failures === 0
            ? CommandHandlerResult::success($summary)
            : CommandHandlerResult::failure("Batch completed with {$failures} failure(s) out of {$total}.", $summary);
    }

    /**
     * @template U of CommandInterface
     * @param CommandHandlerInterface<U> $inner
     * @return class-string<U>
     */
    private function inferExpectedFromInner(CommandHandlerInterface $inner): string
    {
        $handler = $inner::class;
        $marker  = '\\CommandHandler\\';

        $pos = strrpos($handler, $marker);
        if ($pos === false || !str_ends_with($handler, 'CommandHandler')) {
            throw new InvalidArgumentException("Cannot infer expected command from {$handler}");
        }

        $rootNs = substr($handler, 0, $pos);
        $short  = substr($handler, $pos + strlen($marker));
        $base   = substr($short, 0, -strlen('CommandHandler'));

        /** @var class-string<U> $candidate */
        $candidate = $rootNs . '\\Command\\' . $base . 'Command';

        if (!class_exists($candidate)) {
            throw new InvalidArgumentException("Inferred command class {$candidate} not found for inner {$handler}");
        }

        return $candidate;
    }
}
