<?php

declare(strict_types=1);

namespace Makro\Command\CommandHandler;

use InvalidArgumentException;
use LogicException;
use Makro\Command\Contract\CommandGuardInterface;
use Makro\Command\Contract\CommandHandlerGuardInterface;
use Makro\Command\Contract\CommandHandlerInterface;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Guard\NullCommandGuard;
use Makro\Command\Guard\NullCommandHandlerGuard;
use Makro\Command\Result\CommandHandlerResult;

/**
 * @template T of CommandInterface
 * @implements CommandHandlerInterface<T>
 */
abstract class AbstractCommandHandler implements CommandHandlerInterface
{
    protected CommandGuardInterface $commandGuard;

    /**
     * @param CommandHandlerGuardInterface<T>|null $commandHandlerGuard
     * @param CommandGuardInterface|null $commandGuard
     */
    public function __construct(
        ?CommandHandlerGuardInterface $commandHandlerGuard = null,
        ?CommandGuardInterface        $commandGuard = null
    ) {
        ($commandHandlerGuard ?? $this->defaultCommandHandlerGuard())->assert($this);
        $this->commandGuard = $commandGuard ?? $this->defaultCommandGuard();
    }

    protected function defaultCommandHandlerGuard(): CommandHandlerGuardInterface
    {
        // Versuche …\Guard\XyzCommandHandlerGuard, sonst Null
        $fqcn = $this->inferGuardFqcn('CommandHandlerGuard');
        if ($fqcn !== null && class_exists($fqcn) && is_a($fqcn, CommandHandlerGuardInterface::class, true)) {
            return new $fqcn();
        }

        return new NullCommandHandlerGuard();
    }

    protected function defaultCommandGuard(): CommandGuardInterface
    {
        // Versuche …\Guard\XyzCommandGuard, sonst Null
        $fqcn = $this->inferGuardFqcn('CommandGuard');
        if ($fqcn !== null && class_exists($fqcn) && is_a($fqcn, CommandGuardInterface::class, true)) {
            return new $fqcn();
        }

        return new NullCommandGuard();
    }

    final public function handle(CommandInterface $command): CommandHandlerResult
    {
        $this->assertSupportedCommand($command);
        $this->commandGuard->assert($command);

        /** @var T $command */
        return $this->doHandle($command);
    }

    /**
     * @return class-string<T>
     */
    protected function supportedCommand(): string
    {
        $handler = static::class;
        $marker  = '\\CommandHandler\\';

        $pos = strrpos($handler, $marker);
        if ($pos === false || !str_ends_with($handler, 'CommandHandler')) {
            throw new LogicException("Cannot infer supported command for {$handler}: missing {$marker} in FQCN or wrong suffix.");
        }

        $rootNs = substr($handler, 0, $pos);                              // z.B. Makro\CopySurvey
        $short  = substr($handler, $pos + strlen($marker));               // z.B. CopySurveyCommandHandler
        $base   = substr($short, 0, -strlen('CommandHandler'));           // z.B. CopySurvey
        $fqcn   = $rootNs . '\\Command\\' . $base . 'Command';            // z.B. Makro\CopySurvey\Command\CopySurveyCommand

        if (!class_exists($fqcn)) {
            throw new LogicException("Inferred command class {$fqcn} not found for handler {$handler}");
        }

        /** @var class-string<CommandInterface> $fqcn */
        return $fqcn;
    }

    /** @param T $command */
    abstract protected function doHandle(CommandInterface $command): CommandHandlerResult;

    protected function assertSupportedCommand(CommandInterface $command): void
    {
        $expected = $this->supportedCommand();

        if (!class_exists($expected)) {
            throw new LogicException(
                sprintf('supportedCommand() must return class-string of %s; got %s', CommandInterface::class, $expected)
            );
        }

        if (!$command instanceof $expected) {
            throw new InvalidArgumentException(
                sprintf('Unexpected command %s; expected %s', $command::class, $expected)
            );
        }
    }

    private function inferGuardFqcn(string $suffix): ?string
    {
        // Sucht …\Guard\Xyz{$suffix}
        $handler = static::class;
        $marker  = '\\CommandHandler\\';
        $pos = strrpos($handler, $marker);
        if ($pos === false) {
            return null;
        }
        $rootNs = substr($handler, 0, $pos);               // Makro\CopySurvey
        $short  = substr($handler, $pos + strlen($marker)); // CopySurveyCommandHandler

        if (!str_ends_with($short, 'CommandHandler')) {
            return null;
        }
        $base = substr($short, 0, -strlen('CommandHandler')); // CopySurvey

        return $rootNs . '\\Guard\\' . $base . $suffix;       // Makro\CopySurvey\Guard\CopySurveyCommandGuard
    }
}
