# makro/command

Leichtgewichtiges Command/Handler-Kernpaket mit Guards und Basishandler für wiederverwendbare, testbare Anwendungslogik.

## Features

* **Klare Contracts:** `CommandInterface`, `CommandHandlerInterface`, `CommandGuardInterface`, `CommandHandlerGuardInterface`.
* **Ergebnisobjekt:** `CommandHandlerResult` (Factories `success()` / `failure()`).
* **Basishandler:** `AbstractCommandHandler` (template-method, Typ-Check, Guard-Hookpoints).
* **Konvention vor Konfiguration:**
  Handler-FQCN `…\CommandHandler\FooCommandHandler` ⇒

    * Command: `…\FooCommand`
    * Guards (falls vorhanden): `…\Guard\FooCommandGuard` & `…\Guard\FooCommandHandlerGuard`
      Fallback sind **Null-Guards**.
* **Minimale Abhängigkeiten**, PHP **8.0+**.

## Installation

```bash
composer require makro/command
```

> Während der Entwicklung alternativ via VCS/Path-Repository einbinden.

## Anforderungen

* PHP ^8.0
* Composer

## Schnellstart

### 1) Command definieren

```php
<?php declare(strict_types=1);

namespace App\Demo;

use Makro\Command\Contract\CommandInterface;

final class CopySurveyCommand implements CommandInterface
{
    public function __construct(
        public int $sourceSurveyId,
        public ?string $newTitle = null,
    ) {}
}
```

### 2) (Optional) Guards anlegen – werden automatisch gefunden

```php
<?php declare(strict_types=1);

namespace App\Demo\Guard;

use Makro\Command\Contract\CommandGuardInterface;
use Makro\Command\Contract\CommandInterface;
use App\Demo\CopySurveyCommand;

final class CopySurveyCommandGuard implements CommandGuardInterface
{
    /** @param CopySurveyCommand $command */
    public function assert(CommandInterface $command): void
    {
        if ($command->sourceSurveyId <= 0) {
            throw new \InvalidArgumentException('sourceSurveyId must be positive.');
        }
    }
}
```

> Legst du zusätzlich `App\Demo\Guard\CopySurveyCommandHandlerGuard` an, wird dieser Guard im Handler-Konstruktor automatisch ausgeführt (z. B. um Abhängigkeiten/Umgebung zu prüfen).

### 3) Handler implementieren – **nur** `doHandle()` schreiben

```php
<?php declare(strict_types=1);

namespace App\Demo;

use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Result\CommandHandlerResult;

/** @extends AbstractCommandHandler<CopySurveyCommand> */
final class CopySurveyCommandHandler extends AbstractCommandHandler
{
    /** @param CopySurveyCommand $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {
        // Business-Logik …
        return CommandHandlerResult::success(['newSurveyId' => 123]);
    }
}
```

> Dank Konvention brauchst du **keinen** Konstruktor und **kein** `supportedCommand()` mehr.
> Existieren passende Guards im Namespace `App\Demo\Guard`, werden sie automatisch verwendet; sonst greifen Null-Guards.

### 4) Verwendung

```php
$handler = new App\Demo\CopySurveyCommandHandler();
$result  = $handler->handle(new App\Demo\CopySurveyCommand(sourceSurveyId: 42, newTitle: 'Copy of 42'));

if ($result->success) {
    // OK
} else {
    // Fehlerbehandlung per $result->message / $result->data
}
```

## API-Überblick

### Contracts

* **`CommandInterface`** – Marker-Interface für Befehle (Value Objects empfohlen).
* **`CommandHandlerInterface`** – `handle(CommandInterface): CommandHandlerResult`.
* **`CommandGuardInterface`** – Validierung eines konkreten Commands vor Ausführung.
* **`CommandHandlerGuardInterface`** – Invarianten/Preconditions eines Handlers (z. B. Abhängigkeiten da?).

### Ergebnis

* **`CommandHandlerResult`**
  Properties: `bool $success`, `?string $message`, `array $data`
  Factories: `success(array $data = [], ?string $message = null)`, `failure(string $message, array $data = [])`

### Basishandler

* **`Makro\Command\CommandHandler\AbstractCommandHandler`**

    * Finaler Flow in `handle()`:

        1. **Typprüfung** (gegen das abgeleitete/überschriebene Command)
        2. **Command-Guard** (`assert()`)
        3. **`doHandle()`**
    * Du implementierst:

        * `doHandle(CommandInterface $command): CommandHandlerResult`
    * Optional überschreibbar:

        * `supportedCommand(): class-string<T>` (nur wenn du von der Konvention abweichen willst)
        * `defaultCommandGuard()` / `defaultCommandHandlerGuard()` (um Defaults zu beeinflussen)

### Konvention (Namensauflösung)

Ausgehend vom Handler-FQCN `Vendor\Modul\CommandHandler\FooCommandHandler`:

| Abgeleitetes Element | FQCN                                                     |
| -------------------- | -------------------------------------------------------- |
| **Command**          | `Vendor\Modul\FooCommand`                                |
| **Command-Guard**    | `Vendor\Modul\Guard\FooCommandGuard` *(optional)*        |
| **Handler-Guard**    | `Vendor\Modul\Guard\FooCommandHandlerGuard` *(optional)* |

Nicht vorhandene Guards werden durch **Null-Guards** ersetzt.

## Best Practices

* **Commands** als kleine, valide Value Objects (ggf. mit Basiskontrollen im Ctor).
* **Guards** für Domänen- und Umgebungskontrollen (synchron prüfbar).
* **Exceptions** für außergewöhnliche Fehler (Infra/Umgebung); erwartbare Fehlerpfade können über `CommandHandlerResult::failure()` signalisiert werden.
* **Handler schlank** halten; komplexe Teilprozesse in Services/Aktionen auslagern.
* **Static Analysis**: Generics-PHPDoc verwenden (`@template T`, `@extends AbstractCommandHandler<T>`), wie in den Beispielen.

## Tests (Pest)

Minimaltest, der die Konvention nutzt:

```php
<?php

use Makro\Command\Result\CommandHandlerResult;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\CommandHandler\AbstractCommandHandler;

final class PingCommand implements CommandInterface {}

final class PingCommandHandler extends AbstractCommandHandler
{
    /** @param PingCommand $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {
        return CommandHandlerResult::success(['pong' => true]);
    }
}

it('handles ping', function () {
    $result = (new PingCommandHandler())->handle(new PingCommand());
    expect($result->success)->toBeTrue()->and($result->data['pong'])->toBeTrue();
});
```

## Paketstruktur (Empfehlung)

```
src/
  Command/
  CommandHandler/
    AbstractCommandHandler.php
  Contract/
  Guard/
  Result/
```

### Batch-Verarbeitung

Das Paket bietet einen generischen Batch-Flow für homogene Commands.

```php
use Makro\Command\Command\BatchCommand;
use Makro\Command\CommandHandler\BatchCommandHandler;

// Dein Einzelhandler
use Makro\CopySurvey\CommandHandler\CopySurveyCommandHandler;
use Makro\CopySurvey\CopySurveyCommand;

$commands = [
    new CopySurveyCommand(/* … */),
    new CopySurveyCommand(/* … */),
];

$batch   = new BatchCommand($commands, stopOnFirstError: false);
$handler = new BatchCommandHandler(new CopySurveyCommandHandler());

$result = $handler->handle($batch);

if ($result->success) {
    // alle Items erfolgreich
} else {
    // Teilfehler/Fehler — Details in $result->data['items']
    // item: ['index'=>int, 'status'=>'success'|'failure'|'error', 'message'?, 'data'?, 'exception'?, 'code'?]
}
```
