<?php

declare(strict_types=1);

namespace Makro\Tagger\Presentation\Http\RequestHandler\Coder;

use Laminas\Diactoros\Response\HtmlResponse;
use Laminas\Diactoros\Response\RedirectResponse;
use Makro\Tagger\Application\UseCase\SaveCoding\Command\SaveCodingCommand;
use Makro\Tagger\Application\UseCase\SaveCoding\CommandHandler\SaveCodingCommandHandler;
use Makro\Tagger\Domain\Context;
use Makro\Tagger\Domain\ValueObject\Coder;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Makro\Tagger\Infrastructure\Provider\Db\DbAnswerProvider;
use Makro\Tagger\Presentation\Routing\UrlGenerator;
use Makro\Tagger\Presentation\View\TemplateRenderer;
use Psr\Container\ContainerInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Server\RequestHandlerInterface;
use Yii;

final class CodingRequestHandler implements RequestHandlerInterface
{
    public function __construct(
        private ContainerInterface $container,
        private TemplateRenderer $renderer,
        private UrlGenerator $urlGenerator,
    )
    {
    }

    public function handle(ServerRequestInterface $request): ResponseInterface
    {
        /** @var Context $context */
        $context = $request->getAttribute('context');
        /** @var CodingQuestion $question */
        $question = $request->getAttribute('question');
        /** @var Coder $coder */
        $coder = $request->getAttribute('coder');

        if ($request->getMethod() === 'POST') {

            $post = $request->getParsedBody();

            $command = new SaveCodingCommand(
                surveyId: $context->surveyId(),
                responseId: intval($post['responseId']),
                language: $context->language(),
                questionId: $question->questionId(),
                coderId: $coder->id(),
                tag: $post['selectedTag'],
                pii: $post['selectedPii'],
                comment: $post['comment']
            );

           $commandHandler = $this->container->get(SaveCodingCommandHandler::class);
           $result = $commandHandler->handle($command);
           if ($result->success === true) {
               return new RedirectResponse($this->urlGenerator->route('coding', [
                   'sid' => $context->surveyId(),
                   'qid' => $question->questionId(),
               ]));
           }
        }

        $codings = $context->codingCollection()
            ->forQuestion($question)
            ->forCoder($coder);

        $notSubmittedCodings = $codings->notSubmittedBy($coder);

        $totalCodingsCount = $codings->count();
        $notSubmittedCodingsCount = $notSubmittedCodings->count();
        $doneRate = round((($totalCodingsCount - $notSubmittedCodingsCount) / $totalCodingsCount) * 100);

        $currentCoding = $notSubmittedCodings->first();

        if (!$currentCoding) {
            Yii::app()->coder->setFlash('success', 'All codings coded');
            return new RedirectResponse($this->urlGenerator->route('coder-home', [
                'sid' => $context->sid(),
            ]));
        }

        $answerProvider = new DbAnswerProvider($context);
        $answers = $answerProvider->provide($currentCoding);

        $html = $this->renderer->render('coder/coding', [
            'sid' => $context->sid(),
            'qid' => $question->qid(),
            'coder' => $coder,
            'currentCoding' => $currentCoding,
            'tagAnswers' => $answers['tag'],
            'pIIAnswers' => $answers['pII'],
            'totalCodingsCount' => $totalCodingsCount,
            'doneRate' => $doneRate,
            'coderHomeUrl' => $this->urlGenerator->route('coder-home', [
                'sid' => $context->sid(),
            ]),
        ]);

        return new HtmlResponse($html);
    }
}
