<?php

declare(strict_types=1);

namespace Makro\Tagger\Presentation\Http\Middleware;

use Psr\Container\ContainerInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\MiddlewareInterface;
use Psr\Http\Server\RequestHandlerInterface;

final class AuthSelectorMiddleware implements MiddlewareInterface
{
    public function __construct(
        private ContainerInterface $container,
        private array              $authMap
    )
    {
    }

    public function process(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        $action = $request->getQueryParams()['action'] ?? 'home';
        $map = $this->authMap;

        // ermitteln, welche Handlerklasse für diesen action zuständig ist
        $handlers = $map['handlerMap'] ?? [];
        $authMap = $map['authMap'] ?? [];

        $handlerClass = $handlers[$action] ?? null;
        if (!$handlerClass) {
            return $handler->handle($request); // keine Route, kein Auth
        }

        // passende AuthMiddlewares finden
        $middlewares = [];
        foreach ($authMap as $prefix => $mwClasses) {
            if (str_starts_with($handlerClass, $prefix)) {
                foreach ($mwClasses as $mwClass) {
                    $middlewares[] = $this->container->get($mwClass);
                }
            }
        }

        // schichte die AuthMiddlewares ein (innen = original handler)
        $wrappedHandler = $handler;
        foreach (array_reverse($middlewares) as $mw) {
            $wrappedHandler = new class($mw, $wrappedHandler) implements RequestHandlerInterface {
                public function __construct(
                    private MiddlewareInterface     $mw,
                    private RequestHandlerInterface $next
                )
                {
                }

                public function handle(ServerRequestInterface $request): ResponseInterface
                {
                    return $this->mw->process($request, $this->next);
                }
            };
        }

        return $wrappedHandler->handle($request);
    }
}
