<?php

declare(strict_types=1);

namespace Makro\Tagger\Infrastructure\Provider;

use InvalidArgumentException;
use Makro\Tagger\Domain\Collection\CoderCollection;
use Makro\Tagger\Domain\Context;
use Makro\Tagger\Domain\Provider\CoderCollectionProvider;
use Makro\Tagger\Domain\ValueObject\Coder;

class FromStringCoderCollectionProvider implements CoderCollectionProvider
{

    public function provide(Context $context): CoderCollection
    {
        return $this->fromString($context->coders());
    }

    private function fromString(string $coders): CoderCollection
    {
        $result = [];
        $seen   = [];

        $rawLines = preg_split('/\R/u', $coders) ?: [];
        foreach ($rawLines as $idx => $raw) {
            $lineNo = $idx + 1;
            $line   = trim($raw);

            // leer oder Kommentar?
            if ($line === '' || str_starts_with($line, '#') || str_starts_with($line, ';') || str_starts_with($line, '//')) {
                continue;
            }

            // striktes Format: id=>passwort (=> muss vorhanden sein)
            if (strpos($line, '=>') === false) {
                throw new InvalidArgumentException("Ungültige Zeile {$lineNo}: Erwartet Format 'id=>passwort'.");
            }

            [$idRaw, $pwdRaw] = explode('=>', $line, 2);
            $id = trim($idRaw);
            $pwd = trim($pwdRaw);

            if ($id === '') {
                throw new InvalidArgumentException("Ungültige Zeile {$lineNo}: ID fehlt.");
            }
            if ($pwd === '') {
                throw new InvalidArgumentException("Ungültige Zeile {$lineNo}: Passwort darf nicht leer sein.");
            }
            if (isset($seen[$id])) {
                $first = $seen[$id];
                throw new InvalidArgumentException("Duplikat-ID '{$id}' in Zeile {$lineNo} (bereits definiert in Zeile {$first}).");
            }

            $seen[$id] = $lineNo;
            $result[] = new Coder($id, $pwd);
        }

        return new CoderCollection($result);
    }
}
