<?php

declare(strict_types=1);

namespace Makro\Tagger\Infrastructure\Persistence\InMemory;

use InvalidArgumentException;
use Makro\Tagger\Domain\Coding;
use Makro\Tagger\Domain\CodingQuestion;
use Makro\Tagger\Domain\CodingQuestions;
use Makro\Tagger\Domain\CodingRepository;
use Makro\Tagger\Domain\Codings;

/**
 * Einfache In-Memory-Repo für Tests.
 * - Key = (surveyId, responseId, questionId)
 * - assignCoders(): setzt C1/C2 nur, wenn aktuell NULL (simuliert "IS NULL")
 */
class InMemoryCodingRepository implements CodingRepository
{
    /** @var array<string, Coding> */
    private array $store = [];

    /**
     * Optionale Initialdaten.
     * @param iterable<Coding> $codings
     */
    public function __construct(iterable $codings = [])
    {
        foreach ($codings as $coding) {
            $this->put($coding);
        }
    }

    /** Test-Helfer: alle Einträge löschen */
    public function reset(): void
    {
        $this->store = [];
    }

    /** Test-Helfer: einzelne Codings „seeden“ */
    public function seed(Coding ...$codings): void
    {
        foreach ($codings as $c) {
            $this->put($c);
        }
    }

    public function findAll(int $surveyId, CodingQuestions $questions): Codings
    {
        $wantedQids = [];
        foreach ($questions as $q) {
            /** @var CodingQuestion $q */
            $wantedQids[$q->questionId()] = true;
        }

        $result = new Codings();
        foreach ($this->store as $key => $coding) {
            if ($coding->surveyId() !== $surveyId) {
                continue;
            }
            // Nur Codings zu den gewünschten Fragen (falls du das so nutzt)
            $qid = $this->decodeKey($key)['qid'];
            if (!isset($wantedQids[$qid])) {
                continue;
            }
            $result->add($coding);
        }

        return $result;
    }

    public function findOne(int $responseId, CodingQuestion $question): Coding
    {
        $key = $this->key($question->surveyId(), $responseId, $question->questionId());
        if (!isset($this->store[$key])) {
            throw new InvalidArgumentException('Coding not found for given ids');
        }
        return $this->store[$key];
    }

    public function save(Coding $coding): bool
    {
        // „Upsert“: einfach ersetzen
        $this->put($coding);
        return true;
    }

    public function exists(int $responseId, CodingQuestion $question): bool
    {
        return isset($this->store[$this->key($question->surveyId(), $responseId, $question->questionId())]);
    }

    /**
     * Zusätzliche Methode wie in deiner DB-Variante:
     * Setzt C1/C2 aus dem übergebenen Coding, aber nur wenn aktuell NULL.
     * Gibt true zurück, wenn mindestens ein Feld tatsächlich geändert wurde.
     */
    public function assignCoders(Coding $coding): bool
    {
        $key = $this->key($coding->surveyId(), $coding->responseId(), $coding->codingQuestion()->questionId());

        // Anlegen, falls noch nicht vorhanden
        if (!isset($this->store[$key])) {
            $this->store[$key] = $coding;
            // Bei neuem Eintrag gilt die Zuweisung als „erfolgreich“
            return true;
        }

        $stored = $this->store[$key];
        $changed = false;

        $c1New = $coding->getC1();
        $c2New = $coding->getC2();

        if ($c1New === null || $c1New === '') {
            throw new InvalidArgumentException('C1 muss für assignCoders() gesetzt sein');
        }
        if ($c2New !== null && $c2New !== '' && $c1New === $c2New) {
            throw new InvalidArgumentException('C1 und C2 dürfen nicht identisch sein');
        }

        // nur setzen, wenn bisher NULL (simuliert "IS NULL")
        if ($stored->getC1() === null || $stored->getC1() === '') {
            $stored->setC1($c1New);
            $changed = true;
        }

        if ($c2New !== null && $c2New !== '') {
            if ($stored->getC2() === null || $stored->getC2() === '') {
                // Zusätzliche Absicherung: C1 != C2
                if ($stored->getC1() === $c2New) {
                    // Widerspricht der Invariante – in DB würde das über WHERE oder vorherige Logik verhindert.
                    // Hier brechen wir sauber ab.
                    throw new InvalidArgumentException('C2 entspricht C1; Zuweisung abgelehnt');
                }
                $stored->setC2($c2New);
                $changed = true;
            }
        }

        return $changed;
    }

    // --- intern -------------------------------------------------------------

    private function put(Coding $coding): void
    {
        $this->store[$this->key(
            $coding->surveyId(),
            $coding->responseId(),
            $coding->codingQuestion()->questionId()
        )] = $coding;
    }

    private function key(int $sid, int $rid, int $qid): string
    {
        return $sid . ':' . $rid . ':' . $qid;
    }

    /** @return array{sid:int,rid:int,qid:int} */
    private function decodeKey(string $key): array
    {
        [$sid, $rid, $qid] = array_map('intval', explode(':', $key, 3));
        return ['sid' => $sid, 'rid' => $rid, 'qid' => $qid];
    }

    public function findAllForQuestion(CodingQuestion $question): Codings
    {
        // TODO: Implement findAllForQuestion() method.
    }

    public function findOneForQuestion(int $responseId, CodingQuestion $question): Coding
    {
        // TODO: Implement findOneForQuestion() method.
    }
}
