<?php

declare(strict_types=1);

namespace Makro\Tagger\Infrastructure\Persistence\Db;

use InvalidArgumentException;
use Makro\Tagger\Domain\Context;
use Makro\Tagger\Domain\Entity\Coding;
use Makro\Tagger\Domain\Event\C1SubmittedEvent;
use Makro\Tagger\Domain\Event\C2SubmittedEvent;
use Makro\Tagger\Domain\Event\ConflictResolvedEvent;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Makro\Tagger\Domain\Repository\CodingRepository;
use Makro\Tagger\Domain\Collection\CodingCollection;
use Makro\Tagger\Infrastructure\Mapper\Db\DbColumnMapper;
use RuntimeException;
use Yii;

class DbCodingRepository implements CodingRepository
{
    public function find(int $responseId, CodingQuestion $question): ?Coding
    {
        $surveyId = $question->surveyId();
        $tableName = '{{survey_' . $surveyId . '}}';

        $row = Yii::app()->db->createCommand()
            ->select($this->columns($surveyId))
            ->from($tableName)
            ->where('id = :id AND submitdate IS NOT NULL', [':id' => $responseId])
            ->queryRow();

        if (empty($row)) {
            return null;
        }

        return Coding::fromRow($question, $row);
    }

    public function findAll(Context $context): CodingCollection
    {
        $surveyId = $context->surveyId();
        $questions = $context->codingQuestionCollection();

        $tableName = '{{survey_' . $surveyId . '}}';

        $rows = Yii::app()->db->createCommand()
            ->select($this->columns($surveyId))
            ->from($tableName)
            ->where('submitdate IS NOT NULL')
            ->queryAll();

        $codings = new CodingCollection();
        foreach ($rows as $row) {
            foreach ($questions as $question) {
                $codings->add(Coding::fromRow($question, $row));
            }
        }

        return $codings;
    }

    public function findAllForQuestion(CodingQuestion $question): CodingCollection
    {
        $surveyId = $question->surveyId();
        $tableName = '{{survey_' . $surveyId . '}}';

        $rows = Yii::app()->db->createCommand()
            ->select($this->columns($surveyId))
            ->from($tableName)
            ->where('submitdate IS NOT NULL')
            ->queryAll();

        if (!$rows) {
            throw new RuntimeException('Codings not found');
        }

        $codings = new CodingCollection();
        foreach ($rows as $row) {
            $codings->add(Coding::fromRow($question, $row));
        }

        return $codings;
    }

    public function assignCoders(Coding $coding): bool
    {
        $question = $coding->codingQuestion();
        $tableName = '{{survey_' . $question->surveyId() . '}}';

        $columnMapper = new DbColumnMapper($question->surveyId());
        $db = Yii::app()->db;

        $c1 = $coding->getC1();
        $c2 = $coding->getC2();
        $status = $coding->status();

        if ($c1 === null || $c1 === '') {
            throw new InvalidArgumentException('C1 must be set.');
        }

        if ($c2 !== null && $c2 !== '' && $c2 === $c1) {
            throw new InvalidArgumentException('C1 and C2 must not be equal.');
        }

        $c1Col = $columnMapper->column($coding->c1Title());
        $c2Col = $columnMapper->column($coding->c2Title());
        $statusCol = $columnMapper->column($coding->statusTitle());

        $values = [$c1Col => $c1];
        $condParts = ['id = :id'];
        $params = [':id' => $coding->responseId()];

        if ($c2 !== null && $c2 !== '') {
            $values[$c2Col] = $c2;
        }

        $values[$statusCol] = $status;

        $affected = $db->createCommand()->update(
            $tableName,
            $values,
            implode(' AND ', $condParts),
            $params
        );

        return $affected > 0;
    }

    public function save(Coding $coding): void
    {
        $events = $coding->releaseEvents();

        foreach ($events as $event) {
            if ($event instanceof C1SubmittedEvent) {
                $this->persistC1Entry($coding);
            } elseif ($event instanceof C2SubmittedEvent) {
                $this->persistC2Entry($coding);
            } elseif ($event instanceof ConflictResolvedEvent) {
                $this->persistConflictResolution($coding);
            }
        }
    }

    public function exists(int $responseId, CodingQuestion $question): bool
    {
        return true;
    }

    private function persistC1Entry(Coding $coding): void
    {
        $tableName = '{{survey_' . $coding->surveyId() . '}}';

        $mapper = new DbColumnMapper($coding->surveyId());

        Yii::app()->db->createCommand()->update(
            $tableName,
            [
                $mapper->column($coding->c1Title()) => $coding->getC1(),
                $mapper->column($coding->c1TagTitle()) => $coding->getC1Entry()->tag(),
                $mapper->column($coding->c1PIITitle()) => $coding->getC1Entry()->pii(),
                $mapper->column($coding->c1ComTitle()) => $coding->getC1Entry()->comment(),
                $mapper->column($coding->statusTitle()) => $coding->status(),
            ],
            'id = :id',
            [':id' => $coding->responseId()]
        );
    }

    private function persistC2Entry(Coding $coding): void
    {
        $tableName = '{{survey_' . $coding->surveyId() . '}}';

        $mapper = new DbColumnMapper($coding->surveyId());

        Yii::app()->db->createCommand()->update(
            $tableName,
            [
                $mapper->column($coding->c2Title()) => $coding->getC2(),
                $mapper->column($coding->c2TagTitle()) => $coding->getC2Entry()->tag(),
                $mapper->column($coding->c2PIITitle()) => $coding->getC2Entry()->pii(),
                $mapper->column($coding->c2ComTitle()) => $coding->getC2Entry()->comment(),
                $mapper->column($coding->statusTitle()) => $coding->status(),
            ],
            'id = :id',
            [':id' => $coding->responseId()]
        );
    }

    private function persistConflictResolution(Coding $coding): void
    {
        $tableName = '{{survey_' . $coding->surveyId() . '}}';
        Yii::app()->db->createCommand()->update($tableName, [
            $coding->crTitle() => $coding->conflictResolverId(),
            $coding->finalTagTitle() => $coding->finalTag(),
            $coding->finalPIITitle() => $coding->finalPII(),
            $coding->finalComTitle() => $coding->finalCom(),
            $coding->statusTitle() => $coding->status(),
        ], ['id' => $coding->responseId()]);
    }

    private function columns(int $surveyId): array
    {
        $columnMapper = new DbColumnMapper($surveyId);

        $assoc = array_flip($columnMapper->all());
        return array_merge(['id'], array_map(fn($k, $v) => "{$k} as {$v}", array_keys($assoc), $assoc));
    }
}
