<?php

declare(strict_types=1);

namespace Makro\Tagger\Infrastructure\Mapper\Db;

use InvalidArgumentException;
use LimeExpressionManager;
use Makro\Tagger\Domain\Mapper\ColumnMapper;
use Yii;

/**
 * Mappt QuestionTitle (Qcode) -> SGQA-Spaltenname für survey_{sid}.
 * Nutzt LimeExpressionManager::getLEMqcode2sgqa($sid) und cached pro Survey.
 */
final class DbColumnMapper implements ColumnMapper
{
    /** @var array<int, array{exact: array<string,string>, ci: array<string,string>, reverse: array<string,string>}> */
    private static array $cache = [];

    public function __construct(
        private int $surveyId,
    )
    {
        $this->warm();
    }

    private function warm(): void
    {
        if (isset(self::$cache[$this->surveyId])) {
            return;
        }

        $this->ensureLEM();

        // Liefert z.B. ['Q1' => '12345X67X89', 'Q1_SQ001' => '12345X67X89SQ001', ...]
        $map = LimeExpressionManager::getLEMqcode2sgqa($this->surveyId) ?: [];

        // exact map
        $exact = $map;

        // case-insensitive Index (lowercase-Key), verweist auf denselben SGQA
        $ci = [];
        foreach ($map as $qcode => $sgqa) {
            $ci[mb_strtolower($qcode)] = $sgqa;
        }

        // ✅ NEU: Reverse map (SGQA -> Qcode)
        $reverse = array_flip($map);

        self::$cache[$this->surveyId] = [
            'exact' => $exact,
            'ci' => $ci,
            'reverse' => $reverse,
        ];
    }

    /** Gibt den SGQA-Spaltennamen zurück oder null, wenn unbekannt. */
    public function column(string $questionTitle): ?string
    {
        $bucket = self::$cache[$this->surveyId];

        if (isset($bucket['exact'][$questionTitle])) {
            return $bucket['exact'][$questionTitle];
        }

        return null;
    }

    /** Wie column(), wirft aber eine klare Exception wenn nicht vorhanden. */
    public function requireColumn(string $questionTitle): string
    {
        $col = $this->column($questionTitle);
        if ($col === null) {
            throw new InvalidArgumentException(
                "Unbekannter QuestionTitle '{$questionTitle}' für Survey {$this->surveyId}."
            );
        }

        return $col;
    }

    /**
     * ✅ NEU: Gibt den QuestionTitle (Qcode) für einen SGQA-Spaltennamen zurück
     *
     * @param string $columnName Der SGQA-Spaltenname (z.B. '12345X67X89')
     * @return string|null Der QuestionTitle/Qcode (z.B. 'Q1') oder null
     */
    public function questionTitle(string $columnName): ?string
    {
        $bucket = self::$cache[$this->surveyId];

        return $bucket['reverse'][$columnName] ?? null;
    }

    /**
     * ✅ NEU: Wie questionTitle(), wirft aber Exception wenn nicht vorhanden
     *
     * @param string $columnName Der SGQA-Spaltenname
     * @return string Der QuestionTitle/Qcode
     * @throws InvalidArgumentException
     */
    public function requireQuestionTitle(string $columnName): string
    {
        $title = $this->questionTitle($columnName);
        if ($title === null) {
            throw new InvalidArgumentException(
                "Unbekannter Spaltenname '{$columnName}' für Survey {$this->surveyId}."
            );
        }

        return $title;
    }

    /** Gibt die gesamte Map (Qcode -> SGQA) zurück. */
    public function all(): array
    {
        return self::$cache[$this->surveyId]['exact'];
    }

    /**
     * ✅ NEU: Gibt die Reverse-Map (SGQA -> Qcode) zurück
     */
    public function allReverse(): array
    {
        return self::$cache[$this->surveyId]['reverse'];
    }

    /** Existenzcheck für QuestionTitle. */
    public function has(string $questionTitle): bool
    {
        return $this->column($questionTitle) !== null;
    }

    /**
     * ✅ NEU: Existenzcheck für Spaltenname
     */
    public function hasColumn(string $columnName): bool
    {
        return $this->questionTitle($columnName) !== null;
    }

    private function ensureLEM(string $requiredMethod = 'getLEMqcode2sgqa'): void
    {
        // Klasse schon geladen?
        if (!class_exists('LimeExpressionManager', false)) {
            // Versuche, den Helper zu laden (wirft bei Fehlschlag eine Exception)
            Yii::import('application.helpers.replacements_helper', true);
            Yii::import('application.helpers.expressions.em_manager_helper', true);
        }

        if (!class_exists('LimeExpressionManager', false)) {
            throw new \RuntimeException('LimeExpressionManager nicht gefunden (Helper nicht ladbar).');
        }

        if (!method_exists('LimeExpressionManager', $requiredMethod)) {
            throw new \RuntimeException("Methode LimeExpressionManager::{$requiredMethod}() nicht vorhanden.");
        }
    }
}
