<?php

declare(strict_types=1);

namespace Makro\Tagger\Domain\Policy;

use LogicException;
use Makro\Tagger\Domain\Collection\CodingCollection;
use Makro\Tagger\Domain\Collection\CoderCollection;
use Makro\Tagger\Domain\Entity\Coding;

/**
 * Verteilt Codings gleichmäßig auf Coder (C1/C2) nach Round-Robin-Prinzip.
 * - C1 wird allen unassigned Codings gleichmäßig zugewiesen.
 * - Ein konfigurierter Anteil (shareForCoder2) wird zusätzlich C2-zugewiesen.
 */
final class RoundRobinAssignmentPolicy
{
    public function __construct(
        private int $shareForCoder2 = 25, // Prozent, 0–100
    )
    {
        $this->shareForCoder2 = max(0, min(100, $shareForCoder2));
    }

    /**
     * Führt die eigentliche Zuweisung durch.
     *
     * @return array{addedC1:int, addedC2:int, unassigned:int}
     */
    public function assign(CoderCollection $coders, CodingCollection $codings): array
    {
        $codersArr = array_values(iterator_to_array($coders));
        $unassignedList = array_values(iterator_to_array($codings->unassigned()));

        $k = count($codersArr);
        $n = count($unassignedList);

        if ($k === 0) {
            throw new LogicException('Keine Coder verfügbar');
        }

        if ($n === 0) {
            return ['addedC1' => 0, 'addedC2' => 0, 'unassigned' => 0];
        }

        // --- C1 round-robin zuweisen ---
        $c1Idx = 0;
        /** @var Coding $coding */
        foreach ($unassignedList as $coding) {
            $coding->setC1($codersArr[$c1Idx % $k]->id());
            $coding->markAssigned();
            $c1Idx++;
        }

        // --- C2 zuweisen ---
        $m = ($k >= 2) ? (int)round($n * $this->shareForCoder2 / 100) : 0;
        $chosen = $this->evenlySpacedIndices($n, $m);

        $addedC2 = 0;
        $c2Idx = 0;

        foreach ($chosen as $pos) {
            $coding = $unassignedList[$pos];
            $c1 = $coding->getC1();

            for ($off = 0; $off < $k; $off++) {
                $cand = $codersArr[($c2Idx + $off) % $k]->id();
                if ($cand !== $c1) {
                    $coding->setC2($cand);
                    $addedC2++;
                    $c2Idx = ($c2Idx + $off + 1) % $k;
                    break;
                }
            }
        }

        return [
            'addedC1' => $n,
            'addedC2' => $addedC2,
            'unassigned' => $n,
        ];
    }

    /**
     * Liefert m gleichmäßig verteilte Indizes im Bereich [0, n)
     * (Deterministische Auswahl, keine Randomness)
     *
     * @return int[]
     */
    private function evenlySpacedIndices(int $n, int $m): array
    {
        if ($m <= 0) return [];
        if ($m >= $n) return range(0, $n - 1);

        $out = [];
        $acc = 0;
        for ($i = 0; $i < $n; $i++) {
            $acc += $m;
            if ($acc >= $n) {
                $out[] = $i;
                $acc -= $n;
            }
        }

        // @codeCoverageIgnoreStart
        $diff = $m - count($out);
        if ($diff > 0) {
            for ($i = $n - 1; $i >= 0 && $diff > 0; $i--) {
                if (!in_array($i, $out, true)) {
                    $out[] = $i;
                    $diff--;
                }
            }
            sort($out);
        } elseif ($diff < 0) {
            $out = array_slice($out, 0, $m);
        }
        // @codeCoverageIgnoreEnd

        return $out;
    }
}
