<?php

declare(strict_types=1);

namespace Makro\Tagger\Domain\Entity;

use Makro\Tagger\Domain\Event\C1SubmittedEvent;
use Makro\Tagger\Domain\Event\C2SubmittedEvent;
use Makro\Tagger\Domain\Event\ConflictResolvedEvent;
use Makro\Tagger\Domain\ValueObject\Coder;
use Makro\Tagger\Domain\ValueObject\CodingEntry;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Makro\Tagger\Domain\ValueObject\Status;

class Coding
{
    private int $responseId;
    private CodingQuestion $codingQuestion;

    private string $answer;

    private CodingEntry $c1; // Coder 1
    private CodingEntry $c2; // Coder 2

    private ?int $status = null; // Status
    private CodingEntry $cr; // ConflictResolver

    private array $events = [];

    public const SUFFIX_C1 = 'C1';
    public const SUFFIX_C1_TAG = 'C1Tag';
    public const SUFFIX_C1_PII = 'C1PII';
    public const SUFFIX_C1_COM = 'C1Com';

    public const SUFFIX_C2 = 'C2';
    public const SUFFIX_C2_TAG = 'C2Tag';
    public const SUFFIX_C2_PII = 'C2PII';
    public const SUFFIX_C2_COM = 'C2Com';

    public const SUFFIX_STATUS = 'Status';

    public const SUFFIX_CR = 'CR';
    public const SUFFIX_FINAL_TAG = 'FinalTag';
    public const SUFFIX_FINAL_PII = 'FinalPII';
    public const SUFFIX_FINAL_COM = 'FinalCom';

    public function __construct(int $responseId, CodingQuestion $codingQuestion)
    {
        $this->responseId = $responseId;
        $this->codingQuestion = $codingQuestion;
        $this->status = Status::UNASSIGNED;
        $this->c1 = new CodingEntry();
        $this->c2 = new CodingEntry();
        $this->cr = new CodingEntry();
    }

    private function recordEvent(object $event): void
    {
        $this->events[] = $event;
    }

    public function releaseEvents(): array
    {
        $events = $this->events;
        $this->events = [];
        return $events;
    }

    public static function fromRow(CodingQuestion $q, array $row): self
    {
        $coding = new self((int)$row['id'], $q);

        $coding->answer = $row[$coding->answerTitle()];

        $coding->c1 = (new CodingEntry($row[$coding->c1Title()] ?? null))
            ->withSubmission(
                tag: $row[$coding->c1TagTitle()] ?? null,
                pii: $row[$coding->c1PIITitle()] ?? null,
                comment: $row[$coding->c1ComTitle()] ?? null
            );

        $coding->c2 = (new CodingEntry($row[$coding->c2Title()] ?? null))
            ->withSubmission(
                tag: $row[$coding->c2TagTitle()] ?? null,
                pii: $row[$coding->c2PIITitle()] ?? null,
                comment: $row[$coding->c2ComTitle()] ?? null
            );

        $coding->cr = (new CodingEntry($row[$coding->crTitle()] ?? null))
            ->withSubmission(
                tag: $row[$coding->finalTagTitle()] ?? null,
                pii: $row[$coding->finalPIITitle()] ?? null,
                comment: $row[$coding->finalComTitle()] ?? null
            );

        $coding->status = Status::normalize($row[$coding->statusTitle()] ?? null);

        $coding->events = [];

        return $coding;
    }

    public function codingQuestion(): CodingQuestion
    {
        return $this->codingQuestion;
    }

    public function surveyId(): int
    {
        return $this->codingQuestion->surveyId();
    }

    public function responseId(): int
    {
        return $this->responseId;
    }

    public function answer(): string
    {
        return $this->answer;
    }

    public function status(): int
    {
        return $this->status ?? Status::UNASSIGNED;
    }

    public function setStatus(int $status): void
    {
        $this->status = $status;
    }

    public function setC1(string $id): void
    {
        $this->c1 = $this->c1->withCoderId($id);
    }

    public function getC1(): ?string
    {
        return $this->c1->coderId();
    }

    public function getC1Entry(): CodingEntry
    {
        return $this->c1;
    }

    public function getC2Entry(): CodingEntry
    {
        return $this->c2;
    }

    public function setC2(string $id): void
    {
        $this->c2 = $this->c2->withCoderId($id);
    }

    public function getC2(): ?string
    {
        return $this->c2->coderId();
    }

    public function hasC1(): bool
    {
        return $this->c1->isAssigned();
    }

    public function hasC2(): bool
    {
        return $this->c2->isAssigned();
    }

    public function conflictResolverId(): ?string
    {
        return $this->cr->coderId();
    }

    public function finalTag(): ?string
    {
        return $this->cr->tag();
    }

    public function finalPII(): ?string
    {
        return $this->cr->pii();
    }

    public function finalCom(): ?string
    {
        return $this->cr->comment();
    }

    public function markAssigned(): void
    {
        if (!Status::isUnassigned($this->status)) {
            return;
        }

        $this->status = Status::AWAITS_CODING;
    }

    public function submit(Coder $coder, string $tag, string $pii, ?string $comment = null): void
    {
        if ($coder->id() === $this->getC1()) {
            $this->submitC1($tag, $pii, $comment);
        } elseif ($coder->id() === $this->getC2()) {
            $this->submitC2($tag, $pii, $comment);
        }
    }

    public function submitC1(string $tag, string $pii, ?string $comment = null): void
    {
        $this->c1 = $this->c1->withSubmission($tag, $pii, $comment);
        $this->recompute();

        $this->recordEvent(new C1SubmittedEvent(
            codingId: $this->responseId,
            entry: $this->c1,
            surveyId: $this->surveyId(),
            questionId: $this->codingQuestion()->questionId()
        ));
    }

    public function submitC2(string $tag, string $pii, ?string $comment = null): void
    {
        $this->c2 = $this->c2->withSubmission($tag, $pii, $comment);
        $this->recompute();

        $this->recordEvent(new C2SubmittedEvent(
            codingId: $this->responseId,
            entry: $this->c2,
            surveyId: $this->surveyId(),
            questionId: $this->codingQuestion()->questionId()
        ));
    }

    public function hasConflict(): bool
    {
        return $this->c1->conflictsWith($this->c2);
    }

    public function isFinalized(): bool
    {
        return $this->status === Status::DONE;
    }

    public function effectiveFinalTag(): ?string
    {
        if ($this->cr->isSubmitted()) return $this->cr->tag();
        if (!$this->isFinalized()) return null;
        if (!$this->hasC2()) return $this->c1->tag();               // Single
        return $this->c1->tag();                                    // Konsens (== C2)
    }

    public function effectiveFinalPII(): ?string
    {
        if ($this->cr->isSubmitted()) return $this->cr->pii();
        if (!$this->isFinalized()) return null;
        if (!$this->hasC2()) return $this->c1->pii();
        return $this->c1->pii();
    }

    public function effectiveFinalComment(): ?string
    {
        if ($this->cr->isSubmitted()) return $this->cr->comment();
        if (!$this->isFinalized()) return null;
        if (!$this->hasC2()) return $this->c1->comment();
        return $this->c2->comment() ?? $this->c1->comment();        // Konsens: C2 vor C1
    }

    private function recompute(): void
    {
        // Single-coder: kein C2
        if (!$this->hasC2()) {
            if ($this->c1->isSubmitted()) {
                $this->status = Status::DONE;
            } else {
                $this->status = Status::AWAITS_CODING;
            }
            return;
        }

        $c1Ready = $this->c1->isSubmitted();
        $c2Ready = $this->c2->isSubmitted();

        if ($c1Ready && $c2Ready) {
            if ($this->c1->conflictsWith($this->c2)) {
                $this->status = Status::CONFLICTUAL;
            } else {
                $this->status = Status::DONE;
            }
            return;
        }

        // Teil-Ergebnis
        $this->status = Status::AWAITS_CODING;
    }

    public function resolveConflict(string $finalTag, string $finalPII, ?string $finalCom = null, ?string $resolverId = null): void
    {
        if ($this->status !== Status::CONFLICTUAL) {
            throw new \DomainException('Conflict can only be resolved from CONFLICTUAL (3).');
        }

        if (!$this->c1->isSubmitted() || !$this->c2->isSubmitted()) {
            throw new \LogicException('Both coder results must exist to resolve a conflict.');
        }

        // Jetzt erst $cr füllen: finale Werte + wer gelöst hat
        $this->cr = $this->cr
            ->withCoderId($resolverId)
            ->withSubmission($finalTag, $finalPII, $finalCom);

        $this->status = Status::DONE;

        $this->recordEvent(new ConflictResolvedEvent(
            codingId: $this->responseId,
            resolution: $this->cr,
            surveyId: $this->surveyId(),
            questionId: $this->codingQuestion->questionId()
        ));
    }

    public function c1Title(): string
    {
        return $this->title(self::SUFFIX_C1);
    }

    public function c1TagTitle(): string
    {
        return $this->title(self::SUFFIX_C1_TAG);
    }

    public function c1PIITitle(): string
    {
        return $this->title(self::SUFFIX_C1_PII);
    }

    public function c1ComTitle(): string
    {
        return $this->title(self::SUFFIX_C1_COM);
    }

    public function c2Title(): string
    {
        return $this->title(self::SUFFIX_C2);
    }

    public function c2TagTitle(): string
    {
        return $this->title(self::SUFFIX_C2_TAG);
    }

    public function c2PIITitle(): string
    {
        return $this->title(self::SUFFIX_C2_PII);
    }

    public function c2ComTitle(): string
    {
        return $this->title(self::SUFFIX_C2_COM);
    }

    public function statusTitle(): string
    {
        return $this->title(self::SUFFIX_STATUS);
    }

    public function crTitle(): string
    {
        return $this->title(self::SUFFIX_CR);
    }

    public function finalTagTitle(): string
    {
        return $this->title(self::SUFFIX_FINAL_TAG);
    }

    public function finalPIITitle(): string
    {
        return $this->title(self::SUFFIX_FINAL_PII);
    }

    public function finalComTitle(): string
    {
        return $this->title(self::SUFFIX_FINAL_COM);
    }

    public function answerTitle(): string
    {
        return $this->codingQuestion->title();
    }

    public function getTagTitle(Coder $coder): string
    {
        return $coder->id() === $this->getC1() ? $this->c1TagTitle() : $this->c2TagTitle();
    }

    public function getPIITitle(Coder $coder): string
    {
        return $coder->id() === $this->getC1() ? $this->c1PIITitle() : $this->c2PIITitle();
    }

    public function getComTitle(Coder $coder): string
    {
        return $coder->id() === $this->getC1() ? $this->c1ComTitle() : $this->c2ComTitle();
    }

    private function title(string $suffix): string
    {
        return $this->codingQuestion->title() . $suffix;
    }
}
