<?php

declare(strict_types=1);

namespace Makro\Tagger\Domain\Collection;

use ArrayIterator;
use Countable;
use InvalidArgumentException;
use IteratorAggregate;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Traversable;

class CodingQuestionCollection implements IteratorAggregate, Countable
{
    private int $surveyId;
    /** @var array<CodingQuestion> */
    private array $items = [];
    private string $language;

    public function __construct(int $surveyId, array $items = [], string $language = 'en')
    {
        $language = trim($language);
        if ($language === '') {
            throw new InvalidArgumentException('language must not be empty');
        }

        foreach ($items as $item) {
            if (!$item instanceof CodingQuestion) {
                throw new InvalidArgumentException('CodingQuestion allowed only');
            }
            if ($item->surveyId() !== $surveyId) {
                throw new InvalidArgumentException('All items must match given surveyId');
            }
        }

        $this->surveyId = $surveyId;
        $this->items = array_values($items);
        $this->language = $language;
    }

    public function getIterator(): Traversable
    {
        return new ArrayIterator($this->items);
    }

    public function count(): int
    {
        return count($this->items);
    }

    public function surveyId(): int
    {
        return $this->surveyId;
    }

    public function sid(): int
    {
        return $this->surveyId;
    }

    public function language(): string
    {
        return $this->language;
    }

    public function firstWhere(callable $predicate): ?CodingQuestion
    {
        foreach ($this->items as $item) {
            if ($predicate($item)) {
                return $item;
            }
        }

        return null;
    }
}
