<?php

declare(strict_types=1);

namespace Makro\Tagger\Domain\Collection;

use ArrayIterator;
use Countable;
use InvalidArgumentException;
use IteratorAggregate;
use Makro\Tagger\Domain\Entity\Coding;
use Makro\Tagger\Domain\ValueObject\Coder;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Makro\Tagger\Domain\ValueObject\Status;
use Traversable;

class CodingCollection implements IteratorAggregate, Countable
{
    /** @var array<Coding> */
    private array $items = [];

    public function __construct(array $items = [])
    {
        foreach ($items as $item) {
            if (!$item instanceof Coding) {
                throw new InvalidArgumentException('Coding allowed only');
            }
        }

        $this->items = array_values($items);
    }

    public function add(Coding $coding): void
    {
        $this->items[] = $coding;
    }

    public function getIterator(): Traversable
    {
        return new ArrayIterator($this->items);
    }

    public function count(): int
    {
        return count($this->items);
    }

    public function unassigned(): self
    {
        return $this->filter(fn(Coding $c) => Status::isUnassigned($c->status()));
    }

    public function awaiting(): self
    {
        return $this->filter(fn(Coding $c) => Status::awaitsCoding($c->status()));
    }

    public function haveConflicts(): self
    {
        return $this->filter(fn(Coding $c) => Status::hasConflicts($c->status()));
    }

    public function done(): self
    {
        return $this->filter(fn(Coding $c) => Status::isDone($c->status()));
    }

    public function forCoder(Coder $coder): self
    {
        return $this->filter(fn(Coding $c) => $c->getC1() === $coder->id() || $c->getC2() === $coder->id());
    }

    public function submittedBy(Coder $coder): self
    {
        return $this->filter(function(Coding $c) use ($coder) {
            return $c->getC1Entry()->isSubmittedBy($coder)
                || $c->getC2Entry()->isSubmittedBy($coder);
        });
    }

    public function notSubmittedBy(Coder $coder): self
    {
        return $this->filter(function(Coding $c) use ($coder) {
            $c1 = $c->getC1Entry();
            $c2 = $c->getC2Entry();

            // Coder ist C1 und hat noch nicht submitted
            if ($c1->coderId() === $coder->id() && !$c1->isSubmitted()) {
                return true;
            }

            // Coder ist C2 und hat noch nicht submitted
            if ($c2->coderId() === $coder->id() && !$c2->isSubmitted()) {
                return true;
            }

            return false;
        });
    }

    public function forQuestion(CodingQuestion $question): self
    {
        return $this->filter(fn(Coding $c) => $c->codingQuestion()->equals($question));
    }

    public function filter(callable $predicate): static
    {
        $filtered = [];
        foreach ($this->items as $item) {
            if ($predicate($item) === true) {
                $filtered[] = $item;
            }
        }

        return new static($filtered);
    }

    public function limit(int $limit): static
    {
        if ($limit <= 0) {
            return new static([]);
        }

        return new static(array_slice($this->items, 0, $limit));
    }

    public function first(): ?Coding
    {
        return $this->items[0] ?? null;
    }
}
