<?php

declare(strict_types=1);

namespace Makro\Tagger\Application\UseCase\SaveCoding\CommandHandler;

use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Contract\CommandGuardInterface;
use Makro\Command\Contract\CommandHandlerGuardInterface;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Event\NullEventDispatcher;
use Makro\Tagger\Application\UseCase\SaveCoding\Command\SaveCodingCommand;
use Makro\Tagger\Domain\Repository\CodingQuestionRepository;
use Makro\Tagger\Domain\Repository\CodingRepository;
use Makro\Tagger\Domain\ValueObject\Coder;
use Makro\Tagger\Domain\ValueObject\CodingQuestion;
use Psr\EventDispatcher\EventDispatcherInterface;

/** @extends AbstractCommandHandler<SaveCodingCommand> */
class SaveCodingCommandHandler extends AbstractCommandHandler
{
    private EventDispatcherInterface $eventDispatcher;

    public function __construct(
        private CodingRepository      $codingRepository,
        private CodingQuestionRepository $codingQuestionRepository,
        ?EventDispatcherInterface     $eventDispatcher = null,
        ?CommandHandlerGuardInterface $handlerGuard = null,
        ?CommandGuardInterface        $commandGuard = null
    )
    {
        $this->eventDispatcher = $eventDispatcher ?? new NullEventDispatcher();

        parent::__construct(
            commandHandlerGuard: $handlerGuard,
            commandGuard: $commandGuard
        );
    }

    /** @param SaveCodingCommand $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {

        /** @var CodingQuestion $codingQuestion */
        $codingQuestion = $this->codingQuestionRepository->find(
            $command->surveyId(),
            $command->questionId(),
            $command->language()
        );

        if (!$codingQuestion) {
            return CommandHandlerResult::failure(
                message: 'Question not found'
            );
        }

        $coding = $this->codingRepository->find($command->responseId(), $codingQuestion);
        if (!$coding) {
            return CommandHandlerResult::failure(
                message: 'Coding not found'
            );
        }

        $coder = new Coder($command->coderId(), '');

        $coding->submit($coder, $command->tag(), $command->pii(), $command->comment());
        $this->codingRepository->save($coding);

        foreach ($coding->releaseEvents() as $event) {
            $this->eventDispatcher->dispatch($event);
        }

        return CommandHandlerResult::success(
            data: [
                'responseId' => $coding->responseId(),
                'questionId' => $codingQuestion->questionId(),
            ],
            message: 'Coding saved successfully'
        );
    }
}
