<?php

declare(strict_types=1);

namespace Makro\Tagger\Application\UseCase\AssignCoders\CommandHandler;

use Makro\Command\CommandHandler\AbstractCommandHandler;
use Makro\Command\Contract\CommandGuardInterface;
use Makro\Command\Contract\CommandHandlerGuardInterface;
use Makro\Command\Contract\CommandInterface;
use Makro\Command\Result\CommandHandlerResult;
use Makro\Event\NullEventDispatcher;
use Makro\Tagger\Application\UseCase\AssignCoders\Command\AssignCodersCommand;
use Makro\Tagger\Domain\Policy\RoundRobinAssignmentPolicy;
use Makro\Tagger\Domain\Repository\CodingRepository;
use Psr\EventDispatcher\EventDispatcherInterface;

/** @extends AbstractCommandHandler<AssignCodersCommand> */
class AssignCodersCommandHandler extends AbstractCommandHandler
{
    private EventDispatcherInterface $eventDispatcher;

    public function __construct(
        private CodingRepository      $codingRepository,
        ?EventDispatcherInterface     $eventDispatcher = null,
        ?CommandHandlerGuardInterface $handlerGuard = null,
        ?CommandGuardInterface        $commandGuard = null
    )
    {
        $this->eventDispatcher = $eventDispatcher ?? new NullEventDispatcher();

        parent::__construct(
            commandHandlerGuard: $handlerGuard,
            commandGuard: $commandGuard
        );
    }

    /** @param AssignCodersCommand $command */
    protected function doHandle(CommandInterface $command): CommandHandlerResult
    {
        $surveyId = $command->context()->surveyId();
        $sharePercent = $command->context()->shareForCoder2();
        $coders = $command->context()->coderCollection();
        $codings = $command->context()->codingCollection();

        $policy = new RoundRobinAssignmentPolicy($sharePercent);
        $result = $policy->assign($coders, $codings);

        $addedC1 = 0;
        foreach ($codings as $coding) {
            if ($this->codingRepository->assignCoders($coding)) {
              $addedC1++;
            }
        }

        return CommandHandlerResult::success([
            'status' => 'ok',
            'surveyId' => $surveyId,
            'shareForC2' => $sharePercent,
            'unassigned' => $result['unassigned'],
            'addedC1' => $addedC1,
            'addedC2' => $result['addedC2'],
        ], 'C1 vollständig zugewiesen, C2 gemäß Quote ergänzt');
    }
}
