<?php

class Tagger extends PluginBase
{
    protected $storage = 'LimeSurvey\PluginManager\DbStorage';
    protected static $description = 'Tagger';
    protected static $name = 'Tagger';

    /**
     * subscribe to all needed events
     * @see https://manual.limesurvey.org/Plugin_events
     */
    public function init(): void
    {
        require_once __DIR__ . '/vendor/autoload.php';

        /* always first events, when there are survey level settings */
        $this->subscribe('beforeSurveySettings');
        $this->subscribe('newSurveySettings');

        /* from here order events alphabetically */
        $this->subscribe('newDirectRequest');
        $this->subscribe('newQuestionAttributes');
    }

    /**
     * beforeSurveySettings
     * @see https://framagit.org/Shnoulle/exampleSettings/blob/master/exampleSettings.php
     */
    public function beforeSurveySettings(): void
    {
        $event = $this->event;
        $surveyId = intval($event->get('survey'));

        $adminDashboardLink = CHtml::link('AdminDashboard', [
            'plugins/direct',
            'plugin' => 'Tagger',
            'route' => 'admin-index',
            'sid' => $surveyId
        ], [
            'target' => '_blank'
        ]);

        $coderDashboardLink = CHtml::link('CoderDashboard', [
            'plugins/direct',
            'plugin' => 'Tagger',
            'route' => 'admin-login',
            'sid' => $surveyId
        ], [
            'target' => '_blank'
        ]);

        $event->set("surveysettings.{$this->id}", [
            'name' => get_class($this),
            'settings' => [
                'links' => [
                    'type' => 'info',
                    'content' => 'Click here for the ' . $adminDashboardLink . '<br /> Click here for the ' . $coderDashboardLink,
                ],
                'isActive' => [
                    'type' => 'boolean',
                    'label' => 'Active',
                    'current' => $this->getIsActive($surveyId),
                    'help' => 'Activate this plugin for this survey'
                ],
                'coders' => [
                    'type' => 'text',
                    'label' => 'Coders',
                    'current' => $this->getCoders($surveyId),
                    'help' => 'Add a name-password pair line by line separated by "=>", marko.bischof=>123456',
                    'htmlOptions' => [
                        'rows' => 10
                    ]
                ],
                'shareForCoder2' => [
                    'type' => 'int',
                    'label' => 'Share for Coder 2 in %',
                    'current' => $this->getShareForCoder2($surveyId),
                    'help' => 'Enter a percentage value to define for how many responses tagging should be done by two coders.'
                ],
                'info' => [
                    'type' => 'info',
                    'content' => $this->getInfo()
                ],
            ]
        ]);
    }

    public function getIsActive($surveyId): bool
    {
        return (bool)$this->get('isActive', 'Survey', $surveyId, false);
    }

    public function getCoders($surveyId): string
    {
        return $this->get('coders', 'Survey', $surveyId, '');
    }

    public function getShareForCoder2(int $surveyId): int
    {
        return intval($this->get('shareForCoder2', 'Survey', $surveyId, 25));
    }

    public function getVersion(): string
    {
        try {
            if ($this->config === null) {
                $this->readConfigFile();
            }

            if (
                isset($this->config) &&
                isset($this->config->metadata) &&
                isset($this->config->metadata->version)
            ) {
                return (string)$this->config->metadata->version;
            }

            return 'unknown';
        } catch (Exception $e) {
            return 'unknown';
        }

    }

    public function getInfo(): string
    {
        return 'Version: ' . $this->getVersion()
            . ' <br /> Copyright &copy 2015 - ' . date('Y') . ' Survey Consulting <br />';
    }

    public function newSurveySettings(): void
    {
        $event = $this->event;
        foreach ($event->get('settings') as $name => $value) {
            $this->set($name, $value, 'Survey', $event->get('survey'));
        }
    }

    public function newQuestionAttributes(): void
    {
        $attributes = [
            'taggable' => [
                'types' => 'T', // for long text only
                'category' => 'Tagger',
                'sortorder' => 10,
                'inputtype' => 'switch',
                'default' => 0,
                'caption' => 'Activate Tagging',
                'help' => 'Mark this question so it can later be tagged.'
            ]
        ];

        $this->getEvent()->append('questionAttributes', $attributes);

    }

    public function newDirectRequest(): void
    {
        $event = $this->getEvent();
        $target = $event->get('target');
        if (strtolower($target) !== strtolower(get_class($this))) {
            return;
        }

        Yii::app()->setComponent('coder', [
            'class' => 'CWebUser',
            'stateKeyPrefix' => 'coder_'
        ]);

        $container  = (require __DIR__ . '/config/container.php')($this);
        $newDirectRequest = $container->get(\Makro\Tagger\Presentation\Http\NewDirectRequest::class);
        $newDirectRequest->run();

        Yii::app()->end();
    }
}
